<?php
header('Content-Type: application/json');
include 'config.php';

function get_ip_address() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        $ip = $_SERVER['REMOTE_ADDR'];
    }
    return $ip;
}

$settings = [];
$settings_result = $conn->query("SELECT setting_name, setting_value FROM settings WHERE setting_name IN ('app_name', 'from_email', 'email_password')");

if ($settings_result && $settings_result->num_rows > 0) {
    while ($row = $settings_result->fetch_assoc()) {
        $settings[$row['setting_name']] = $row['setting_value'];
    }
}

if (!isset($settings['app_name']) || !isset($settings['from_email']) || !isset($settings['email_password'])) {
    echo json_encode(['status' => 'error', 'message' => 'Email settings are not configured in the database.']);
    $conn->close();
    exit;
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (empty(trim($_POST['email']))) {
        echo json_encode(['status' => 'error', 'message' => 'Email is required.']);
        exit;
    }

    $email = filter_var(trim($_POST['email']), FILTER_SANITIZE_EMAIL);

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid email format.']);
        exit;
    }

    $ip_address = get_ip_address();
    
    $stmt_rate = $conn->prepare("SELECT COUNT(*) as count, MAX(request_time) as latest_request, NOW() as mysql_now 
                                FROM ip_log 
                                WHERE ip_address = ? AND request_time >= (NOW() - INTERVAL 1 HOUR)");
    $stmt_rate->bind_param("s", $ip_address);
    $stmt_rate->execute();
    $rate_result = $stmt_rate->get_result()->fetch_assoc();
    $stmt_rate->close();

    $request_count = $rate_result['count'];
    $latest_request_time = strtotime($rate_result['latest_request'] ?? '1970-01-01 00:00:00');
    $current_time = strtotime($rate_result['mysql_now']);
    
    $wait_time_seconds = 0;

    if ($request_count == 2) {
        $wait_time_seconds = 3 * 60;
    } elseif ($request_count == 3) {
        $wait_time_seconds = 10 * 60;
    } elseif ($request_count > 3) {
        $wait_time_seconds = 30 * 60;
    }

    if ($wait_time_seconds > 0) {
        $time_elapsed = $current_time - $latest_request_time;
        if ($time_elapsed < $wait_time_seconds) {
            $time_remaining = $wait_time_seconds - $time_elapsed;
            $minutes_remaining = ceil($time_remaining / 60);
            echo json_encode(['status' => 'error', 'message' => "Too many requests. Please wait {$minutes_remaining} minutes before trying again."]);
            $conn->close();
            exit;
        }
    }
    
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows == 0) {
        echo json_encode(['status' => 'error', 'message' => 'Email address not found.']);
        $stmt->close();
        $conn->close();
        exit;
    }
    
    $stmt->close();

    try {
        $token = bin2hex(random_bytes(32));
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => 'Could not generate token.']);
        exit;
    }
    
    $stmt = $conn->prepare("UPDATE users SET rs_auth_token = ?, token_expired = (NOW() + INTERVAL 1 HOUR) WHERE email = ?");
    $stmt->bind_param("ss", $token, $email);

    if ($stmt->execute()) {
        
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
        $host = $_SERVER['HTTP_HOST'];
        
        $directory_path = dirname($_SERVER['PHP_SELF']);
        $directory = rtrim(str_replace(' ', '%20', $directory_path), '/\\');
        if ($directory == "" || $directory == ".") { $directory = ""; }

        $reset_url = $protocol . "://" . $host . $directory . "/rs_change_pass.php?rs_auth=" . $token;

        $queryParams = http_build_query([
            'app_name' => $settings['app_name'],
            'from_email' => $settings['from_email'],
            'password' => $settings['email_password'],
            'to_email' => $email,
            'reset_url' => $reset_url
        ]);

        $mail_api_url = $protocol . "://" . $host . $directory . "/rs_mail.php?" . $queryParams;

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $mail_api_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $mail_response_json = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            echo json_encode(['status' => 'error', 'message' => 'Failed to call mail API: ' . curl_error($ch), 'debug_info' => ['url' => $mail_api_url]]);
            curl_close($ch);
            exit;
        }
        curl_close($ch);

        $mail_response = json_decode($mail_response_json, true);

        if ($http_code == 200 && isset($mail_response['status']) && $mail_response['status'] == 'success') {
            
            $stmt_log = $conn->prepare("INSERT INTO ip_log (ip_address) VALUES (?)");
            $stmt_log->bind_param("s", $ip_address);
            $stmt_log->execute();
            $stmt_log->close();

            echo json_encode(['status' => 'success', 'message' => 'Password reset link has been sent to your email.']);
        } else {
            $error_msg = isset($mail_response['message']) ? $mail_response['message'] : 'Failed to send email.';
            echo json_encode([
                'status' => 'error', 
                'message' => $error_msg, 
                'mail_api_http_code' => $http_code,
                'debug_info' => [
                    'url' => $mail_api_url,
                    'response' => $mail_response_json
                ]
            ]);
        }

    } else {
        echo json_encode(['status' => 'error', 'message' => 'Failed to update password reset token.']);
    }

    $stmt->close();
    $conn->close();

} else {
    echo json_encode(['status' => 'error', 'message' => 'Invalid request method.']);
}
?>